import { Controller, Get, Param, UseGuards, NotFoundException, Query } from '@nestjs/common';
import { JwtAuthGuard } from '../auth/jwt-auth.guard';
import { UsersService } from './users.service';

@Controller('users')
@UseGuards(JwtAuthGuard)
export class UsersController {
  constructor(private readonly usersService: UsersService) {}

  @Get(':id')
  async getUserById(@Param('id') id: string) {
    const user = await this.usersService.findById(id);
    if (!user) {
      throw new NotFoundException('User not found');
    }
    // Remove sensitive information
    const { passwordHash, ...result } = user;
    return result;
  }

  @Get('companies/search')
  async searchCompanies(@Query('query') query: string) {
    if (!query || query.trim().length < 2) {
      return [];
    }
    
    const companies = await this.usersService.searchCompanies(query);
    return companies.map(({ passwordHash, ...company }) => company);
  }

  @Get()
  async findUsers(@Query('email') email: string, @Query('search') search: string) {
    if (email) {
      const user = await this.usersService.findByEmail(email);
      if (!user) {
        throw new NotFoundException('User not found');
      }
      const { passwordHash, ...result } = user;
      return result;
    }
    
    if (search) {
      const users = await this.usersService.searchByName(search);
      return users.map(({ passwordHash, ...user }) => user);
    }
    
    throw new NotFoundException('Email or search parameter is required');
  }
}