import { Body, Controller, Get, Param, Patch, Req, UploadedFile, UseGuards, UseInterceptors } from '@nestjs/common';
import { JwtAuthGuard } from '../auth/jwt-auth.guard';
import { ProfileService } from './profile.service';
import { FileInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { extname } from 'path';
import { existsSync, mkdirSync } from 'fs';
import type { Express, Request } from 'express';

type FilenameCallback = (error: Error | null, filename: string) => void;

// Ensure uploads directory exists
const uploadsDir = './uploads/avatars';
if (!existsSync(uploadsDir)) {
  mkdirSync(uploadsDir, { recursive: true });
}

@Controller('profile')
@UseGuards(JwtAuthGuard)
export class ProfileController {
  constructor(private readonly profileService: ProfileService) {}

  @Get(':userId')
  getProfile(@Param('userId') userId: string) {
    return this.profileService.getProfileWithStats(userId);
  }

  @Get(':userId/stats')
  getProfileStats(@Param('userId') userId: string) {
    return this.profileService.getProfileStats(userId);
  }

  @Patch(':userId')
  updateProfile(@Param('userId') userId: string, @Body() body: any) {
    return this.profileService.upsert(userId, body);
  }

  // Test endpoint to debug what's being received
  @Patch(':userId/avatar-test')
  async testAvatarUpload(
    @Param('userId') userId: string,
    @Req() req: Request,
    @Body() body: any
  ) {
    console.log('=== AVATAR TEST REQUEST ===');
    console.log('User ID:', userId);
    console.log('Headers:', req.headers);
    console.log('Body:', body);
    console.log('Raw body:', (req as any).rawBody);
    console.log('===========================');
    
    return { message: 'Test endpoint reached', userId, headers: req.headers };
  }

  @Patch(':userId/avatar')
  @UseInterceptors(
    FileInterceptor('avatar', {
      storage: diskStorage({
        destination: './uploads/avatars',
        filename: (_req: Request, file: Express.Multer.File, cb: FilenameCallback) => {
          const unique = Date.now() + '-' + Math.round(Math.random() * 1e9);
          
          // Get extension from originalname or mimetype
          let ext = extname(file.originalname);
          
          // If no extension from filename, get from mimetype
          if (!ext && file.mimetype) {
            const mimeToExt: { [key: string]: string } = {
              'image/jpeg': '.jpg',
              'image/jpg': '.jpg',
              'image/png': '.png',
              'image/gif': '.gif',
              'image/webp': '.webp',
            };
            ext = mimeToExt[file.mimetype] || '.jpg';
          }
          
          // Fallback to .jpg if still no extension
          if (!ext) {
            ext = '.jpg';
          }
          
          cb(null, unique + ext);
        }
      }),
      fileFilter: (_req: Request, file: Express.Multer.File, cb: any) => {
        // Accept images only
        if (!file.mimetype.match(/\/(jpg|jpeg|png|gif|webp)$/)) {
          return cb(new Error('Only image files are allowed!'), false);
        }
        cb(null, true);
      }
    })
  )
  async uploadAvatar(
    @Param('userId') userId: string,
    @UploadedFile() file: Express.Multer.File,
    @Req() req: Request
  ) {
    console.log('=== AVATAR UPLOAD REQUEST ===');
    console.log('User ID:', userId);
    console.log('Content-Type:', req.headers['content-type']);
    console.log('File received:', file);
    console.log('Request body:', req.body);
    console.log('Request files:', (req as any).files);
    console.log('=============================');
    
    if (!file) {
      console.error('No file in request!');
      throw new Error('No file uploaded. Please ensure you are sending a file with the key "avatar"');
    }

    return this.profileService.updateAvatar(userId, file.filename);
  }
}


