import { Injectable, NotFoundException, ForbiddenException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { PostEntity } from './post.entity';
import { LikesEntity } from './likes.entity';
import { CommentsEntity } from './comments.entity';
import { UserEntity } from '../users/user.entity';

@Injectable()
export class PostsService {
  constructor(
    @InjectRepository(PostEntity)
    private readonly postRepo: Repository<PostEntity>,
    @InjectRepository(LikesEntity)
    private readonly likeRepo: Repository<LikesEntity>,
    @InjectRepository(CommentsEntity)
    private readonly commentRepo: Repository<CommentsEntity>,
  ) {}

  // 🔹 Get all posts
  findAll() {
    return this.postRepo.find({
      order: { createdAt: 'DESC' },
      relations: ['author', 'likes', 'comments', 'comments.author'],
    });
  }

  // 🔹 Get feed posts (filtered by followed users)
  async getFeed(userId: string, followedUserIds?: string[]) {
    const query = this.postRepo.createQueryBuilder('post')
      .leftJoinAndSelect('post.author', 'author')
      .leftJoinAndSelect('post.likes', 'likes')
      .leftJoinAndSelect('post.comments', 'comments')
      .leftJoinAndSelect('comments.author', 'commentAuthor')
      .orderBy('post.createdAt', 'DESC');

    if (followedUserIds && followedUserIds.length > 0) {
      query.where('author.id IN (:...userIds)', { userIds: [userId, ...followedUserIds] });
    }

    return query.getMany();
  }

  // 🔹 Get a single post
  async findOne(id: string) {
    const post = await this.postRepo.findOne({
      where: { id },
      relations: ['author', 'likes', 'comments', 'comments.author'],
    });
    if (!post) throw new NotFoundException('Post not found');
    return post;
  }

  // 🔹 Create a post
  create(data: Partial<PostEntity>) {
    const entity = this.postRepo.create(data);
    return this.postRepo.save(entity);
  }

  // 🔹 Create post with image
  async createWithImage(data: Partial<PostEntity>, imageUrl?: string) {
    const postData: Partial<PostEntity> = { ...data, imageUrl };
    const entity = this.postRepo.create(postData);
    const savedPost = await this.postRepo.save(entity);
    
    // Reload with author information
    return this.postRepo.findOne({
      where: { id: savedPost.id },
      relations: ['author', 'likes', 'comments'],
    });
  }

  // 🔹 Verify post ownership
  async verifyPostOwnership(postId: string, userId: string) {
    const post = await this.findOne(postId);
    if (!post) throw new NotFoundException('Post not found');
    if (post.author.id !== userId) {
      throw new ForbiddenException('You can only edit your own posts');
    }
    return post;
  }

  // 🔹 Update a post
  async update(id: string, data: Partial<PostEntity>, userId?: string) {
    const post = await this.findOne(id);
    if (!post) throw new NotFoundException('Post not found');
    
    // Verify ownership if userId is provided
    if (userId && post.author.id !== userId) {
      throw new ForbiddenException('You can only edit your own posts');
    }
    
    Object.assign(post, data);
    return this.postRepo.save(post);
  }

  // 🔹 Delete a post
  remove(id: string) {
    return this.postRepo.delete(id);
  }

  // ==============================
  // ❤️ LIKE METHODS
  // ==============================

  async getLikesByPost(postId: string) {
    return this.likeRepo.find({
      where: { post: { id: postId } },
      relations: ['user', 'post'],
    });
  }

  async addLike(postId: string, userId: string) {
    const post = await this.findOne(postId);
    if (!post) throw new NotFoundException('Post not found');

    const existing = await this.likeRepo.findOne({
      where: { post: { id: postId }, user: { id: userId } },
    });
    if (existing) return existing; // prevent duplicate likes

    const like = this.likeRepo.create({
      post,
      user: { id: userId } as UserEntity,
    });

    await this.likeRepo.save(like);
    return { message: 'Post liked', likeId: like.id };
  }

  async removeLike(postId: string, userId: string) {
    const existing = await this.likeRepo.findOne({
      where: { post: { id: postId }, user: { id: userId } },
    });
    if (!existing) throw new NotFoundException('Like not found');

    await this.likeRepo.delete(existing.id);
    return { message: 'Post unliked' };
  }

  // ==============================
  // 💬 COMMENT METHODS
  // ==============================

  async addComment(postId: string, userId: string, content: string) {
    const post = await this.findOne(postId);
    if (!post) throw new NotFoundException('Post not found');

    const comment = this.commentRepo.create({
      post,
      author: { id: userId } as UserEntity,
      content,
    });

    await this.commentRepo.save(comment);
    return comment;
  }

  async updateComment(commentId: string, userId: string, content: string) {
    const comment = await this.commentRepo.findOne({
      where: { id: commentId },
      relations: ['author'],
    });
    if (!comment) throw new NotFoundException('Comment not found');
    if (comment.author.id !== userId) throw new ForbiddenException('Not your comment');

    comment.content = content;
    await this.commentRepo.save(comment);
    return { message: 'Comment updated', comment };
  }

  async deleteComment(commentId: string, userId: string) {
    const comment = await this.commentRepo.findOne({
      where: { id: commentId },
      relations: ['author'],
    });
    if (!comment) throw new NotFoundException('Comment not found');
    if (comment.author.id !== userId) throw new ForbiddenException('Not your comment');

    await this.commentRepo.delete(comment.id);
    return { message: 'Comment deleted' };
  }

  async getCommentsByPost(postId: string) {
    return this.commentRepo.find({
      where: { post: { id: postId } },
      relations: ['author', 'post'],
      order: { createdAt: 'DESC' },
    });
  }
}
