import { Body, Controller, Delete, Get, Param, Patch, Post, Query, UploadedFile, UseGuards, UseInterceptors } from '@nestjs/common';
import { JwtAuthGuard } from '../auth/jwt-auth.guard';
import { NewsService } from './news.service';
import { FileInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { extname } from 'path';
import type { Express, Request } from 'express';

type FilenameCallback = (error: Error | null, filename: string) => void;

@Controller('news')
@UseGuards(JwtAuthGuard)
export class NewsController {
  constructor(private readonly newsService: NewsService) {}

  @Get()
  getNews(@Query() _filters: any) {
    return this.newsService.findAll();
  }

  @Get(':id')
  getOne(@Param('id') id: string) {
    return this.newsService.findOne(id);
  }

  @Post()
  create(@Body() body: any) {
    return this.newsService.create(body);
  }

  @Post('upload')
  @UseInterceptors(
    FileInterceptor('image', {
      storage: diskStorage({
        destination: 'uploads/news',
        filename: (_req: Request, file: Express.Multer.File, cb: FilenameCallback) => {
          const unique = Date.now() + '-' + Math.round(Math.random() * 1e9);
          cb(null, unique + extname(file.originalname));
        }
      })
    })
  )
  upload(@UploadedFile() file: Express.Multer.File) {
    return { imageUrl: `/uploads/news/${file.filename}` };
  }

  @Patch(':id')
  update(@Param('id') id: string, @Body() body: any) {
    return this.newsService.update(id, body);
  }

  @Delete(':id')
  remove(@Param('id') id: string) {
    return this.newsService.remove(id);
  }
}


