import { Test, TestingModule } from '@nestjs/testing';
import { getRepositoryToken } from '@nestjs/typeorm';
import { NetworkService } from './network.service';
import { ConnectionEntity } from './connection.entity';
import { FollowerEntity } from './follower.entity';
import * as fc from 'fast-check';

/**
 * Feature: maritime-app-enhancements
 * Property 21: Follow creates relationship and increments count
 * Property 22: Unfollow removes relationship and decrements count
 * Validates: Requirements 8.2, 8.3
 */

describe('NetworkService Property Tests', () => {
  let service: NetworkService;
  let followerRepo: any;

  beforeEach(async () => {
    const mockConnectionRepo = {
      find: jest.fn(),
      findOne: jest.fn(),
      create: jest.fn(),
      save: jest.fn(),
      delete: jest.fn()
    };

    const mockFollowerRepo = {
      find: jest.fn(),
      findOne: jest.fn(),
      create: jest.fn((data: any) => data),
      save: jest.fn((data: any) => Promise.resolve({ id: 'follow-id', ...data })),
      delete: jest.fn(),
      count: jest.fn()
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        NetworkService,
        {
          provide: getRepositoryToken(ConnectionEntity),
          useValue: mockConnectionRepo
        },
        {
          provide: getRepositoryToken(FollowerEntity),
          useValue: mockFollowerRepo
        }
      ]
    }).compile();

    service = module.get<NetworkService>(NetworkService);
    followerRepo = module.get(getRepositoryToken(FollowerEntity));
  });

  describe('Property 21: Follow creates relationship and increments count', () => {
    it('should create follow relationship when following a user', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.uuid(),
          fc.uuid(),
          async (followerId, followingId) => {
            fc.pre(followerId !== followingId);

            followerRepo.findOne.mockResolvedValue(null);
            followerRepo.count.mockResolvedValueOnce(5).mockResolvedValueOnce(6);

            const countBefore = await service.getFollowerCount(followingId);
            await service.followUser(followerId, followingId);
            const countAfter = await service.getFollowerCount(followingId);

            expect(followerRepo.save).toHaveBeenCalled();
            expect(countAfter).toBe(countBefore + 1);
          }
        ),
        { numRuns: 50 }
      );
    });
  });

  describe('Property 22: Unfollow removes relationship and decrements count', () => {
    it('should remove follow relationship when unfollowing a user', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.uuid(),
          fc.uuid(),
          async (followerId, followingId) => {
            fc.pre(followerId !== followingId);

            followerRepo.findOne.mockResolvedValue({ id: 'follow-id' });
            followerRepo.count.mockResolvedValueOnce(6).mockResolvedValueOnce(5);

            const countBefore = await service.getFollowerCount(followingId);
            await service.unfollowUser(followerId, followingId);
            const countAfter = await service.getFollowerCount(followingId);

            expect(followerRepo.delete).toHaveBeenCalled();
            expect(countAfter).toBe(countBefore - 1);
          }
        ),
        { numRuns: 50 }
      );
    });
  });
});
