import * as fc from 'fast-check';

/**
 * Feature: job-salary-enhancement, Property 1: Salary range validation
 * Validates: Requirements 1.2
 */

describe('Property 1: Salary range validation', () => {
  // Utility function to validate salary range
  const validateSalaryRange = (minSalary?: number, maxSalary?: number): boolean => {
    // If both values are provided, min should be <= max
    if (minSalary !== undefined && maxSalary !== undefined) {
      return minSalary <= maxSalary;
    }
    // If only one value is provided, it's valid
    // If neither is provided, it's valid (optional fields)
    return true;
  };

  it('should validate that minSalary is less than or equal to maxSalary when both are provided', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: 0, max: 1000000, noNaN: true }),
        fc.float({ min: 0, max: 1000000, noNaN: true }),
        (salary1, salary2) => {
          const minSalary = Math.min(salary1, salary2);
          const maxSalary = Math.max(salary1, salary2);
          
          // When both salaries are provided and min <= max, validation should pass
          const isValid = validateSalaryRange(minSalary, maxSalary);
          expect(isValid).toBe(true);
          
          // When min > max, validation should fail
          if (minSalary !== maxSalary) {
            const isInvalid = validateSalaryRange(maxSalary, minSalary);
            expect(isInvalid).toBe(false);
          }
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should accept valid salary ranges with only minSalary provided', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: 0, max: 1000000, noNaN: true }),
        (minSalary) => {
          const isValid = validateSalaryRange(minSalary, undefined);
          expect(isValid).toBe(true);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should accept valid salary ranges with only maxSalary provided', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: 0, max: 1000000, noNaN: true }),
        (maxSalary) => {
          const isValid = validateSalaryRange(undefined, maxSalary);
          expect(isValid).toBe(true);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should accept when neither salary is provided (optional fields)', async () => {
    const isValid = validateSalaryRange(undefined, undefined);
    expect(isValid).toBe(true);
  });
});