import * as fc from 'fast-check';

/**
 * Feature: job-salary-enhancement, Property 4: Salary display formatting completeness
 * Validates: Requirements 2.1
 */

describe('Property 4: Salary display formatting completeness', () => {
  // Supported currency codes and duration types
  const supportedCurrencies = ['USD', 'EUR', 'GBP', 'CAD', 'AUD', 'JPY', 'CHF', 'SEK', 'NOK', 'DKK'];
  const supportedDurations = ['hourly', 'daily', 'weekly', 'monthly', 'annually'];

  // Currency symbols mapping
  const currencySymbols: Record<string, string> = {
    'USD': '$',
    'EUR': '€',
    'GBP': '£',
    'CAD': 'C$',
    'AUD': 'A$',
    'JPY': '¥',
    'CHF': 'CHF',
    'SEK': 'kr',
    'NOK': 'kr',
    'DKK': 'kr'
  };

  // Duration text mapping
  const durationText: Record<string, string> = {
    'hourly': 'per hour',
    'daily': 'per day',
    'weekly': 'per week',
    'monthly': 'per month',
    'annually': 'per year'
  };

  // Utility function to format salary display
  const formatSalaryDisplay = (minSalary: number, maxSalary: number, currency: string, duration: string): string => {
    const symbol = currencySymbols[currency] || currency;
    const durationStr = durationText[duration] || duration;
    
    // Format numbers with appropriate decimal places
    const formatNumber = (num: number): string => {
      if (currency === 'JPY') {
        // Japanese Yen typically doesn't use decimal places
        return Math.round(num).toLocaleString();
      }
      return num.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: 2 });
    };

    const minFormatted = formatNumber(minSalary);
    const maxFormatted = formatNumber(maxSalary);

    if (minSalary === maxSalary) {
      return `${symbol}${minFormatted} ${durationStr}`;
    } else {
      return `${symbol}${minFormatted} - ${symbol}${maxFormatted} ${durationStr}`;
    }
  };

  it('should include currency symbol, salary range, and duration text for complete salary information', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: 1, max: 500000, noNaN: true }),
        fc.float({ min: 1, max: 500000, noNaN: true }),
        fc.constantFrom(...supportedCurrencies),
        fc.constantFrom(...supportedDurations),
        (salary1, salary2, currency, duration) => {
          const minSalary = Math.min(salary1, salary2);
          const maxSalary = Math.max(salary1, salary2);
          
          // When complete salary information is provided
          const displayString = formatSalaryDisplay(minSalary, maxSalary, currency, duration);
          
          // Then the display should include currency symbol
          const expectedSymbol = currencySymbols[currency] || currency;
          expect(displayString).toContain(expectedSymbol);
          
          // And should include duration text
          const expectedDuration = durationText[duration] || duration;
          expect(displayString).toContain(expectedDuration);
          
          // And should include salary values
          expect(displayString).toMatch(/\d/); // Should contain at least one digit
          
          // And should be a non-empty string
          expect(displayString.length).toBeGreaterThan(0);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should format salary ranges correctly when min and max are different', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: 1, max: 100000, noNaN: true }),
        fc.float({ min: 100001, max: 500000, noNaN: true }), // Ensure max > min
        fc.constantFrom(...supportedCurrencies),
        fc.constantFrom(...supportedDurations),
        (minSalary, maxSalary, currency, duration) => {
          // When min and max salaries are different
          const displayString = formatSalaryDisplay(minSalary, maxSalary, currency, duration);
          
          // Then the display should contain a range separator
          expect(displayString).toMatch(/-/);
          
          // And should contain both salary values
          const symbol = currencySymbols[currency] || currency;
          const minStr = currency === 'JPY' ? Math.round(minSalary).toLocaleString() : minSalary.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: 2 });
          const maxStr = currency === 'JPY' ? Math.round(maxSalary).toLocaleString() : maxSalary.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: 2 });
          
          expect(displayString).toContain(minStr);
          expect(displayString).toContain(maxStr);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should format single salary value when min equals max', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: 1, max: 500000, noNaN: true }),
        fc.constantFrom(...supportedCurrencies),
        fc.constantFrom(...supportedDurations),
        (salary, currency, duration) => {
          // When min and max salaries are the same
          const displayString = formatSalaryDisplay(salary, salary, currency, duration);
          
          // Then the display should not contain a range separator
          expect(displayString).not.toMatch(/-/);
          
          // And should contain the salary value only once in the main part
          const symbol = currencySymbols[currency] || currency;
          const salaryStr = currency === 'JPY' ? Math.round(salary).toLocaleString() : salary.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: 2 });
          
          // Count occurrences of the formatted salary number
          const occurrences = (displayString.match(new RegExp(salaryStr.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'g')) || []).length;
          expect(occurrences).toBe(1);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should handle Japanese Yen formatting without decimal places', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: 1, max: 500000, noNaN: true }),
        fc.float({ min: 1, max: 500000, noNaN: true }),
        fc.constantFrom(...supportedDurations),
        (salary1, salary2, duration) => {
          const minSalary = Math.min(salary1, salary2);
          const maxSalary = Math.max(salary1, salary2);
          
          // When currency is JPY
          const displayString = formatSalaryDisplay(minSalary, maxSalary, 'JPY', duration);
          
          // Then the display should not contain decimal points for JPY
          expect(displayString).not.toMatch(/¥\d+\.\d+/);
          
          // And should contain the yen symbol
          expect(displayString).toContain('¥');
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should produce consistent formatting for the same input', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: 1, max: 500000, noNaN: true }),
        fc.float({ min: 1, max: 500000, noNaN: true }),
        fc.constantFrom(...supportedCurrencies),
        fc.constantFrom(...supportedDurations),
        (salary1, salary2, currency, duration) => {
          const minSalary = Math.min(salary1, salary2);
          const maxSalary = Math.max(salary1, salary2);
          
          // When the same salary information is formatted multiple times
          const display1 = formatSalaryDisplay(minSalary, maxSalary, currency, duration);
          const display2 = formatSalaryDisplay(minSalary, maxSalary, currency, duration);
          
          // Then the results should be identical
          expect(display1).toBe(display2);
        }
      ),
      { numRuns: 100 }
    );
  });
});