import * as fc from 'fast-check';

/**
 * Feature: job-salary-enhancement, Property 6: Multi-job formatting consistency
 * Validates: Requirements 2.4
 */

describe('Property 6: Multi-job formatting consistency', () => {
  // Supported currency codes and duration types
  const supportedCurrencies = ['USD', 'EUR', 'GBP', 'CAD', 'AUD', 'JPY', 'CHF', 'SEK', 'NOK', 'DKK'];
  const supportedDurations = ['hourly', 'daily', 'weekly', 'monthly', 'annually'];

  // Currency symbols mapping
  const currencySymbols: Record<string, string> = {
    'USD': '$',
    'EUR': '€',
    'GBP': '£',
    'CAD': 'C$',
    'AUD': 'A$',
    'JPY': '¥',
    'CHF': 'CHF',
    'SEK': 'kr',
    'NOK': 'kr',
    'DKK': 'kr'
  };

  // Duration text mapping
  const durationText: Record<string, string> = {
    'hourly': 'per hour',
    'daily': 'per day',
    'weekly': 'per week',
    'monthly': 'per month',
    'annually': 'per year'
  };

  // Utility function to format salary display (matches the service implementation)
  const formatSalaryDisplay = (minSalary?: number, maxSalary?: number, currency?: string, duration?: string): string | null => {
    if (!minSalary && !maxSalary) {
      return null;
    }

    if (!currency || !duration) {
      return null;
    }

    const symbol = currencySymbols[currency] || currency;
    const durationStr = durationText[duration] || duration;

    const formatNumber = (num: number): string => {
      if (currency === 'JPY') {
        return Math.round(num).toLocaleString();
      }
      return num.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: 2 });
    };

    if (minSalary && maxSalary) {
      if (minSalary === maxSalary) {
        return `${symbol}${formatNumber(minSalary)} ${durationStr}`;
      } else {
        return `${symbol}${formatNumber(minSalary)} - ${symbol}${formatNumber(maxSalary)} ${durationStr}`;
      }
    } else if (minSalary) {
      return `From ${symbol}${formatNumber(minSalary)} ${durationStr}`;
    } else if (maxSalary) {
      return `Up to ${symbol}${formatNumber(maxSalary)} ${durationStr}`;
    }

    return null;
  };

  // Mock job data generator
  const jobGenerator = fc.record({
    id: fc.uuid(),
    title: fc.string({ minLength: 1, maxLength: 100 }),
    minSalary: fc.option(fc.float({ min: 1, max: 500000, noNaN: true }), { nil: undefined }),
    maxSalary: fc.option(fc.float({ min: 1, max: 500000, noNaN: true }), { nil: undefined }),
    currency: fc.option(fc.constantFrom(...supportedCurrencies), { nil: undefined }),
    salaryDuration: fc.option(fc.constantFrom(...supportedDurations), { nil: undefined })
  }).map(job => {
    // Ensure salary range is valid if both min and max are provided
    if (job.minSalary && job.maxSalary && job.minSalary > job.maxSalary) {
      return { ...job, minSalary: job.maxSalary, maxSalary: job.minSalary };
    }
    return job;
  });

  it('should use consistent formatting rules across all jobs in a list', async () => {
    await fc.assert(
      fc.property(
        fc.array(jobGenerator, { minLength: 2, maxLength: 10 }),
        (jobs) => {
          // When multiple jobs are formatted for display
          const formattedJobs = jobs.map(job => ({
            ...job,
            formattedSalary: formatSalaryDisplay(job.minSalary, job.maxSalary, job.currency, job.salaryDuration)
          }));

          // Then all jobs with the same currency should use the same currency symbol
          const jobsByCurrency = formattedJobs.reduce((acc, job) => {
            if (job.currency && job.formattedSalary) {
              if (!acc[job.currency]) acc[job.currency] = [];
              acc[job.currency].push(job);
            }
            return acc;
          }, {} as Record<string, typeof formattedJobs>);

          Object.entries(jobsByCurrency).forEach(([currency, currencyJobs]) => {
            const expectedSymbol = currencySymbols[currency] || currency;
            currencyJobs.forEach(job => {
              expect(job.formattedSalary).toContain(expectedSymbol);
            });
          });

          // And all jobs with the same duration should use the same duration text
          const jobsByDuration = formattedJobs.reduce((acc, job) => {
            if (job.salaryDuration && job.formattedSalary) {
              if (!acc[job.salaryDuration]) acc[job.salaryDuration] = [];
              acc[job.salaryDuration].push(job);
            }
            return acc;
          }, {} as Record<string, typeof formattedJobs>);

          Object.entries(jobsByDuration).forEach(([duration, durationJobs]) => {
            const expectedDurationText = durationText[duration] || duration;
            durationJobs.forEach(job => {
              expect(job.formattedSalary).toContain(expectedDurationText);
            });
          });
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should format salary ranges consistently regardless of job order', async () => {
    await fc.assert(
      fc.property(
        fc.array(jobGenerator, { minLength: 2, maxLength: 5 }),
        (jobs) => {
          // When the same jobs are formatted in different orders
          const shuffledJobs = [...jobs].sort(() => Math.random() - 0.5);
          
          const originalFormatted = jobs.map(job => 
            formatSalaryDisplay(job.minSalary, job.maxSalary, job.currency, job.salaryDuration)
          );
          
          const shuffledFormatted = shuffledJobs.map(job => 
            formatSalaryDisplay(job.minSalary, job.maxSalary, job.currency, job.salaryDuration)
          );

          // Then jobs with identical salary data should have identical formatting
          jobs.forEach((originalJob, index) => {
            const shuffledIndex = shuffledJobs.findIndex(shuffledJob => 
              shuffledJob.id === originalJob.id
            );
            
            if (shuffledIndex !== -1) {
              expect(originalFormatted[index]).toBe(shuffledFormatted[shuffledIndex]);
            }
          });
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should handle jobs without salary information consistently', async () => {
    await fc.assert(
      fc.property(
        fc.array(fc.record({
          id: fc.uuid(),
          title: fc.string({ minLength: 1, maxLength: 100 }),
          minSalary: fc.constant(undefined),
          maxSalary: fc.constant(undefined),
          currency: fc.constant(undefined),
          salaryDuration: fc.constant(undefined)
        }), { minLength: 1, maxLength: 5 }),
        (jobsWithoutSalary) => {
          // When jobs without salary information are formatted
          const formattedJobs = jobsWithoutSalary.map(job => ({
            ...job,
            formattedSalary: formatSalaryDisplay(job.minSalary, job.maxSalary, job.currency, job.salaryDuration)
          }));

          // Then all should consistently return null for formatted salary
          formattedJobs.forEach(job => {
            expect(job.formattedSalary).toBeNull();
          });
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should format identical salary data identically across different jobs', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: 1, max: 500000, noNaN: true }),
        fc.float({ min: 1, max: 500000, noNaN: true }),
        fc.constantFrom(...supportedCurrencies),
        fc.constantFrom(...supportedDurations),
        fc.integer({ min: 2, max: 5 }),
        (salary1, salary2, currency, duration, jobCount) => {
          const minSalary = Math.min(salary1, salary2);
          const maxSalary = Math.max(salary1, salary2);

          // When multiple jobs have identical salary information
          const jobs = Array.from({ length: jobCount }, (_, index) => ({
            id: `job-${index}`,
            title: `Job ${index}`,
            minSalary,
            maxSalary,
            currency,
            salaryDuration: duration
          }));

          const formattedSalaries = jobs.map(job => 
            formatSalaryDisplay(job.minSalary, job.maxSalary, job.currency, job.salaryDuration)
          );

          // Then all formatted salaries should be identical
          const firstFormatted = formattedSalaries[0];
          formattedSalaries.forEach(formatted => {
            expect(formatted).toBe(firstFormatted);
          });

          // And should not be null since all required fields are provided
          expect(firstFormatted).not.toBeNull();
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should maintain formatting consistency for mixed complete and incomplete salary data', async () => {
    await fc.assert(
      fc.property(
        fc.array(fc.oneof(
          // Complete salary data
          fc.record({
            id: fc.uuid(),
            minSalary: fc.float({ min: 1, max: 500000, noNaN: true }),
            maxSalary: fc.float({ min: 1, max: 500000, noNaN: true }),
            currency: fc.constantFrom(...supportedCurrencies),
            salaryDuration: fc.constantFrom(...supportedDurations)
          }).map(job => ({
            ...job,
            minSalary: Math.min(job.minSalary, job.maxSalary),
            maxSalary: Math.max(job.minSalary, job.maxSalary)
          })),
          // Incomplete salary data
          fc.record({
            id: fc.uuid(),
            minSalary: fc.constant(undefined),
            maxSalary: fc.constant(undefined),
            currency: fc.constant(undefined),
            salaryDuration: fc.constant(undefined)
          })
        ), { minLength: 2, maxLength: 8 }),
        (mixedJobs) => {
          // When a list contains both complete and incomplete salary data
          const formattedJobs = mixedJobs.map(job => ({
            ...job,
            formattedSalary: formatSalaryDisplay(job.minSalary, job.maxSalary, job.currency, job.salaryDuration)
          }));

          // Then complete salary data should be consistently formatted
          const completeJobs = formattedJobs.filter(job => job.formattedSalary !== null);
          const incompleteJobs = formattedJobs.filter(job => job.formattedSalary === null);

          // All complete jobs should have non-null formatted salary
          completeJobs.forEach(job => {
            expect(job.formattedSalary).not.toBeNull();
            expect(typeof job.formattedSalary).toBe('string');
            expect(job.formattedSalary!.length).toBeGreaterThan(0);
          });

          // All incomplete jobs should have null formatted salary
          incompleteJobs.forEach(job => {
            expect(job.formattedSalary).toBeNull();
          });
        }
      ),
      { numRuns: 100 }
    );
  });
});