import { Injectable, NotFoundException, BadRequestException, InternalServerErrorException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, Like } from 'typeorm';
import { JobEntity } from './job.entity';
import { ApplicationEntity } from './application.entity';
import { UserEntity } from '../users/user.entity';
import { CreateJobDto } from './dto/create-job.dto';
import { UpdateJobDto } from './dto/update-job.dto';

@Injectable()
export class JobsService {
  constructor(
    @InjectRepository(JobEntity)
    private readonly jobRepo: Repository<JobEntity>,
    @InjectRepository(ApplicationEntity)
    private readonly applicationRepo: Repository<ApplicationEntity>,
    @InjectRepository(UserEntity)
    private readonly userRepo: Repository<UserEntity>
  ) {}

  // Job methods
  async findAll(filters?: { search?: string; location?: string; jobType?: string; companyId?: string }) {
    const queryBuilder = this.jobRepo.createQueryBuilder('job')
      .leftJoinAndSelect('job.company', 'company')
      .leftJoinAndSelect('job.applications', 'applications')
      .where('job.isActive = :isActive', { isActive: true });

    // Search by title, description, or company name
    if (filters?.search) {
      queryBuilder.andWhere(
        '(LOWER(job.title) LIKE LOWER(:search) OR LOWER(job.description) LIKE LOWER(:search) OR LOWER(company.name) LIKE LOWER(:search))',
        { search: `%${filters.search}%` }
      );
    }

    // Filter by location
    if (filters?.location) {
      queryBuilder.andWhere('LOWER(job.location) LIKE LOWER(:location)', { 
        location: `%${filters.location}%` 
      });
    }

    // Filter by job type
    if (filters?.jobType) {
      queryBuilder.andWhere('job.jobType = :jobType', { jobType: filters.jobType });
    }

    // Filter by company ID (for company users viewing their own jobs)
    if (filters?.companyId) {
      queryBuilder.andWhere('company.id = :companyId', { companyId: filters.companyId });
    }

    const jobs = await queryBuilder
      .orderBy('job.postedDate', 'DESC')
      .getMany();

    return jobs.map(job => this.formatJobResponse(job));
  }

  async findOne(id: string) {
    const job = await this.jobRepo.findOne({ where: { id }, relations: ['company', 'applications'] });
    return job ? this.formatJobResponse(job) : null;
  }

  async create(dto: CreateJobDto, companyId: string) {
    try {
      // Validate required fields
      if (!dto.title || !dto.description || !dto.location) {
        throw new BadRequestException({
          message: 'Validation failed',
          errors: {
            title: !dto.title ? 'Title is required' : undefined,
            description: !dto.description ? 'Description is required' : undefined,
            location: !dto.location ? 'Location is required' : undefined,
          }
        });
      }

      // Validate salary range if both min and max are provided
      if (dto.minSalary !== undefined && dto.maxSalary !== undefined && dto.minSalary > dto.maxSalary) {
        throw new BadRequestException({
          message: 'Validation failed',
          errors: {
            salary: 'Minimum salary cannot be greater than maximum salary'
          }
        });
      }

      // Validate and normalize jobType
      const validJobTypes = ['full-time', 'part-time', 'contract', 'internship'];
      const normalizedJobType = dto.jobType?.toLowerCase();
      
      if (!normalizedJobType || !validJobTypes.includes(normalizedJobType)) {
        throw new BadRequestException({
          message: 'Validation failed',
          errors: {
            jobType: `Job type must be one of: ${validJobTypes.join(', ')}`
          }
        });
      }

      // Find company user
      const company = await this.userRepo.findOne({ where: { id: companyId } });
      if (!company) {
        throw new NotFoundException('Company not found');
      }

      // Map frontend fields to backend entity fields
      const jobEntity = this.jobRepo.create({
        title: dto.title,
        description: dto.description,
        location: dto.location,
        // Handle new salary fields
        minSalary: dto.minSalary,
        maxSalary: dto.maxSalary,
        currency: dto.currency,
        salaryDuration: dto.salaryDuration,
        // Keep backward compatibility with legacy salary field
        salaryRange: dto.salary || this.formatSalaryDisplay(dto.minSalary, dto.maxSalary, dto.currency, dto.salaryDuration) || 'Not specified',
        jobType: normalizedJobType,
        requirements: dto.requirements || '',
        company: company,
        isActive: dto.isActive !== undefined ? dto.isActive : true,
      });

      // Save and return complete job object with generated ID
      const savedJob = await this.jobRepo.save(jobEntity);
      
      // Fetch the complete job with relations and format salary for response
      const job = await this.jobRepo.findOne({ 
        where: { id: savedJob.id },
        relations: ['company']
      });

      return this.formatJobResponse(job);
    } catch (error) {
      // Re-throw known errors
      if (error instanceof BadRequestException || error instanceof NotFoundException) {
        throw error;
      }
      
      // Handle database errors
      throw new InternalServerErrorException({
        message: 'Failed to create job',
        error: error instanceof Error ? error.message : 'Unknown error'
      });
    }
  }

  async update(id: string, updateJobDto: UpdateJobDto) {
    // Validate salary range if both min and max are provided
    if (updateJobDto.minSalary !== undefined && updateJobDto.maxSalary !== undefined && updateJobDto.minSalary > updateJobDto.maxSalary) {
      throw new BadRequestException({
        message: 'Validation failed',
        errors: {
          salary: 'Minimum salary cannot be greater than maximum salary'
        }
      });
    }

    // Convert DTO to entity data, excluding fields that need special handling
    const { company, salary, ...dtoData } = updateJobDto;
    
    const updateData: Partial<JobEntity> = {
      ...dtoData
    };

    // Handle salary fields - update structured salary and maintain backward compatibility
    if (salary !== undefined) {
      updateData.salaryRange = salary;
    } else if (updateJobDto.minSalary !== undefined || updateJobDto.maxSalary !== undefined || 
               updateJobDto.currency !== undefined || updateJobDto.salaryDuration !== undefined) {
      // If any structured salary field is being updated, regenerate the display string
      const currentJob = await this.findOne(id);
      if (currentJob) {
        const newMinSalary = updateJobDto.minSalary !== undefined ? updateJobDto.minSalary : currentJob.minSalary;
        const newMaxSalary = updateJobDto.maxSalary !== undefined ? updateJobDto.maxSalary : currentJob.maxSalary;
        const newCurrency = updateJobDto.currency !== undefined ? updateJobDto.currency : currentJob.currency;
        const newDuration = updateJobDto.salaryDuration !== undefined ? updateJobDto.salaryDuration : currentJob.salaryDuration;
        
        updateData.salaryRange = this.formatSalaryDisplay(newMinSalary, newMaxSalary, newCurrency, newDuration) || 'Not specified';
      }
    }

    // Handle company field separately if provided
    if (company) {
      const companyEntity = await this.userRepo.findOne({ where: { id: company } });
      if (companyEntity) {
        updateData.company = companyEntity;
      }
    }

    await this.jobRepo.update(id, updateData);
    const updatedJob = await this.findOne(id);
    return this.formatJobResponse(updatedJob);
  }

  remove(id: string) {
    return this.jobRepo.delete(id);
  }

  // Application methods
  async applyToJob(jobId: string, applicantId: string, cvUrl: string, coverLetter?: string) {
    const job = await this.findOne(jobId);
    if (!job) {
      throw new NotFoundException('Job not found');
    }

    const existingApplication = await this.applicationRepo.findOne({
      where: {
        job: { id: jobId },
        applicant: { id: applicantId }
      }
    });

    if (existingApplication) {
      throw new BadRequestException('Already applied to this job');
    }

    const application = this.applicationRepo.create({
      job: { id: jobId } as JobEntity,
      applicant: { id: applicantId } as UserEntity,
      cvUrl,
      coverLetter,
      status: 'pending'
    });

    return this.applicationRepo.save(application);
  }

  async getUserApplications(userId: string) {
    return this.applicationRepo.find({
      where: { applicant: { id: userId } },
      relations: ['job'],
      order: { appliedDate: 'DESC' }
    });
  }

  async getJobApplications(jobId: string) {
    return this.applicationRepo.find({
      where: { job: { id: jobId } },
      order: { appliedDate: 'DESC' }
    });
  }

  async updateApplicationStatus(
    applicationId: string,
    status: 'pending' | 'reviewed' | 'accepted' | 'rejected'
  ) {
    const application = await this.applicationRepo.findOne({
      where: { id: applicationId }
    });

    if (!application) {
      throw new NotFoundException('Application not found');
    }

    application.status = status;
    return this.applicationRepo.save(application);
  }

  // Salary formatting utilities
  private formatSalaryDisplay(minSalary?: number, maxSalary?: number, currency?: string, duration?: string): string | null {
    if (!minSalary && !maxSalary) {
      return null;
    }

    if (!currency || !duration) {
      return null;
    }

    const currencySymbols: Record<string, string> = {
      // Major Global Currencies
      'USD': '$',
      'EUR': '€',
      'GBP': '£',
      'JPY': '¥',
      'CHF': 'CHF',
      // North American Currencies
      'CAD': 'C$',
      'MXN': '$',
      // Oceania Currencies
      'AUD': 'A$',
      'NZD': 'NZ$',
      // Nordic Currencies
      'SEK': 'kr',
      'NOK': 'kr',
      'DKK': 'kr',
      // African Currencies
      'ZAR': 'R',
      'NGN': '₦',
      'EGP': '£',
      'KES': 'KSh',
      'GHS': '₵',
      'MAD': 'DH',
      'TND': 'د.ت',
      'ETB': 'Br',
      'UGX': 'USh',
      'TZS': 'TSh',
      'BWP': 'P',
      'MUR': '₨',
      'XOF': 'CFA',
      'XAF': 'FCFA',
      // Asian Currencies
      'CNY': '¥',
      'INR': '₹',
      'KRW': '₩',
      'SGD': 'S$',
      'HKD': 'HK$',
      'THB': '฿',
      'MYR': 'RM',
      'IDR': 'Rp',
      'PHP': '₱',
      'VND': '₫',
      // Middle Eastern Currencies
      'AED': 'د.إ',
      'SAR': '﷼',
      'QAR': '﷼',
      'KWD': 'د.ك',
      'BHD': '.د.ب',
      'OMR': '﷼',
      'ILS': '₪',
      'TRY': '₺',
      // South American Currencies
      'BRL': 'R$',
      'ARS': '$',
      'CLP': '$',
      'COP': '$',
      'PEN': 'S/',
      // Eastern European Currencies
      'PLN': 'zł',
      'CZK': 'Kč',
      'HUF': 'Ft',
      'RON': 'lei',
      'RUB': '₽',
      'UAH': '₴'
    };

    const durationText: Record<string, string> = {
      'hourly': 'per hour',
      'daily': 'per day',
      'weekly': 'per week',
      'monthly': 'per month',
      'annually': 'per year'
    };

    const symbol = currencySymbols[currency] || currency;
    const durationStr = durationText[duration] || duration;

    const formatNumber = (num: number): string => {
      if (currency === 'JPY') {
        return Math.round(num).toLocaleString();
      }
      return num.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: 2 });
    };

    if (minSalary && maxSalary) {
      if (minSalary === maxSalary) {
        return `${symbol}${formatNumber(minSalary)} ${durationStr}`;
      } else {
        return `${symbol}${formatNumber(minSalary)} - ${symbol}${formatNumber(maxSalary)} ${durationStr}`;
      }
    } else if (minSalary) {
      return `From ${symbol}${formatNumber(minSalary)} ${durationStr}`;
    } else if (maxSalary) {
      return `Up to ${symbol}${formatNumber(maxSalary)} ${durationStr}`;
    }

    return null;
  }

  private formatCurrency(amount: number, currencyCode: string): string {
    const currencyRules: Record<string, {
      symbol: string;
      decimals: number;
      symbolPosition: 'before' | 'after';
    }> = {
      // Major Global Currencies
      'USD': { symbol: '$', decimals: 2, symbolPosition: 'before' },
      'EUR': { symbol: '€', decimals: 2, symbolPosition: 'before' },
      'GBP': { symbol: '£', decimals: 2, symbolPosition: 'before' },
      'JPY': { symbol: '¥', decimals: 0, symbolPosition: 'before' },
      'CHF': { symbol: 'CHF', decimals: 2, symbolPosition: 'before' },
      // North American Currencies
      'CAD': { symbol: 'C$', decimals: 2, symbolPosition: 'before' },
      'MXN': { symbol: '$', decimals: 2, symbolPosition: 'before' },
      // Oceania Currencies
      'AUD': { symbol: 'A$', decimals: 2, symbolPosition: 'before' },
      'NZD': { symbol: 'NZ$', decimals: 2, symbolPosition: 'before' },
      // Nordic Currencies
      'SEK': { symbol: 'kr', decimals: 2, symbolPosition: 'after' },
      'NOK': { symbol: 'kr', decimals: 2, symbolPosition: 'after' },
      'DKK': { symbol: 'kr', decimals: 2, symbolPosition: 'after' },
      // African Currencies
      'ZAR': { symbol: 'R', decimals: 2, symbolPosition: 'before' },
      'NGN': { symbol: '₦', decimals: 2, symbolPosition: 'before' },
      'EGP': { symbol: '£', decimals: 2, symbolPosition: 'before' },
      'KES': { symbol: 'KSh', decimals: 2, symbolPosition: 'before' },
      'GHS': { symbol: '₵', decimals: 2, symbolPosition: 'before' },
      'MAD': { symbol: 'DH', decimals: 2, symbolPosition: 'after' },
      'TND': { symbol: 'د.ت', decimals: 3, symbolPosition: 'before' },
      'ETB': { symbol: 'Br', decimals: 2, symbolPosition: 'before' },
      'UGX': { symbol: 'USh', decimals: 0, symbolPosition: 'before' },
      'TZS': { symbol: 'TSh', decimals: 0, symbolPosition: 'before' },
      'BWP': { symbol: 'P', decimals: 2, symbolPosition: 'before' },
      'MUR': { symbol: '₨', decimals: 2, symbolPosition: 'before' },
      'XOF': { symbol: 'CFA', decimals: 0, symbolPosition: 'after' },
      'XAF': { symbol: 'FCFA', decimals: 0, symbolPosition: 'after' },
      // Asian Currencies
      'CNY': { symbol: '¥', decimals: 2, symbolPosition: 'before' },
      'INR': { symbol: '₹', decimals: 2, symbolPosition: 'before' },
      'KRW': { symbol: '₩', decimals: 0, symbolPosition: 'before' },
      'SGD': { symbol: 'S$', decimals: 2, symbolPosition: 'before' },
      'HKD': { symbol: 'HK$', decimals: 2, symbolPosition: 'before' },
      'THB': { symbol: '฿', decimals: 2, symbolPosition: 'before' },
      'MYR': { symbol: 'RM', decimals: 2, symbolPosition: 'before' },
      'IDR': { symbol: 'Rp', decimals: 0, symbolPosition: 'before' },
      'PHP': { symbol: '₱', decimals: 2, symbolPosition: 'before' },
      'VND': { symbol: '₫', decimals: 0, symbolPosition: 'after' },
      // Middle Eastern Currencies
      'AED': { symbol: 'د.إ', decimals: 2, symbolPosition: 'after' },
      'SAR': { symbol: '﷼', decimals: 2, symbolPosition: 'before' },
      'QAR': { symbol: '﷼', decimals: 2, symbolPosition: 'before' },
      'KWD': { symbol: 'د.ك', decimals: 3, symbolPosition: 'before' },
      'BHD': { symbol: '.د.ب', decimals: 3, symbolPosition: 'before' },
      'OMR': { symbol: '﷼', decimals: 3, symbolPosition: 'before' },
      'ILS': { symbol: '₪', decimals: 2, symbolPosition: 'before' },
      'TRY': { symbol: '₺', decimals: 2, symbolPosition: 'before' },
      // South American Currencies
      'BRL': { symbol: 'R$', decimals: 2, symbolPosition: 'before' },
      'ARS': { symbol: '$', decimals: 2, symbolPosition: 'before' },
      'CLP': { symbol: '$', decimals: 0, symbolPosition: 'before' },
      'COP': { symbol: '$', decimals: 0, symbolPosition: 'before' },
      'PEN': { symbol: 'S/', decimals: 2, symbolPosition: 'before' },
      // Eastern European Currencies
      'PLN': { symbol: 'zł', decimals: 2, symbolPosition: 'after' },
      'CZK': { symbol: 'Kč', decimals: 2, symbolPosition: 'after' },
      'HUF': { symbol: 'Ft', decimals: 0, symbolPosition: 'after' },
      'RON': { symbol: 'lei', decimals: 2, symbolPosition: 'after' },
      'RUB': { symbol: '₽', decimals: 2, symbolPosition: 'after' },
      'UAH': { symbol: '₴', decimals: 2, symbolPosition: 'before' }
    };

    const rules = currencyRules[currencyCode];
    if (!rules) {
      throw new Error(`Unsupported currency: ${currencyCode}`);
    }

    const formattedNumber = amount.toLocaleString('en-US', {
      minimumFractionDigits: rules.decimals,
      maximumFractionDigits: rules.decimals,
      useGrouping: true
    });

    if (rules.symbolPosition === 'before') {
      return `${rules.symbol}${formattedNumber}`;
    } else {
      return `${formattedNumber} ${rules.symbol}`;
    }
  }

  private formatJobResponse(job: JobEntity | null): any {
    if (!job) {
      return null;
    }

    return {
      ...job,
      // Add formatted salary information for API responses
      formattedSalary: this.formatSalaryDisplay(job.minSalary, job.maxSalary, job.currency, job.salaryDuration),
      // Keep backward compatibility
      salaryRange: job.salaryRange || this.formatSalaryDisplay(job.minSalary, job.maxSalary, job.currency, job.salaryDuration) || 'Not specified'
    };
  }
}


