import * as fc from 'fast-check';

/**
 * Feature: job-salary-enhancement, Property 8: Duration support completeness
 * Validates: Requirements 3.2
 */

describe('Property 8: Duration support completeness', () => {
  // Supported duration types as specified in requirements
  const supportedDurations = ['hourly', 'daily', 'weekly', 'monthly', 'annually'];
  
  // Duration text mapping (matches the service implementation)
  const durationText: Record<string, string> = {
    'hourly': 'per hour',
    'daily': 'per day',
    'weekly': 'per week',
    'monthly': 'per month',
    'annually': 'per year'
  };

  // Utility function to format salary display with duration (matches the service implementation)
  const formatSalaryWithDuration = (minSalary: number, maxSalary: number, currency: string, duration: string): string => {
    const currencySymbols: Record<string, string> = {
      'USD': '$',
      'EUR': '€',
      'GBP': '£',
      'CAD': 'C$',
      'AUD': 'A$',
      'JPY': '¥',
      'CHF': 'CHF',
      'SEK': 'kr',
      'NOK': 'kr',
      'DKK': 'kr'
    };

    const symbol = currencySymbols[currency] || currency;
    const durationStr = durationText[duration] || duration;

    const formatNumber = (num: number): string => {
      if (currency === 'JPY') {
        return Math.round(num).toLocaleString();
      }
      return num.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: 2 });
    };

    if (minSalary === maxSalary) {
      return `${symbol}${formatNumber(minSalary)} ${durationStr}`;
    } else {
      return `${symbol}${formatNumber(minSalary)} - ${symbol}${formatNumber(maxSalary)} ${durationStr}`;
    }
  };

  // Utility function to validate duration type (simulates backend validation)
  const validateDurationType = (duration: string): boolean => {
    return supportedDurations.includes(duration);
  };

  it('should accept and process all supported duration types correctly', async () => {
    await fc.assert(
      fc.property(
        fc.constantFrom(...supportedDurations),
        fc.float({ min: 1, max: 100000, noNaN: true }),
        fc.constantFrom('USD', 'EUR', 'GBP'), // Use a subset of currencies for simplicity
        (duration, salary, currency) => {
          // When any supported duration type is used
          // Then the system should accept and process it without errors
          expect(validateDurationType(duration)).toBe(true);
          
          const formatted = formatSalaryWithDuration(salary, salary, currency, duration);
          
          // And should produce a non-empty formatted string
          expect(formatted).toBeTruthy();
          expect(typeof formatted).toBe('string');
          expect(formatted.length).toBeGreaterThan(0);
          
          // And should contain the expected duration text
          const expectedDurationText = durationText[duration];
          expect(formatted).toContain(expectedDurationText);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should maintain consistent duration text mapping across all supported durations', async () => {
    await fc.assert(
      fc.property(
        fc.constantFrom(...supportedDurations),
        fc.array(fc.float({ min: 1, max: 50000, noNaN: true }), { minLength: 2, maxLength: 5 }),
        fc.constantFrom('USD', 'EUR', 'GBP'),
        (duration, salaries, currency) => {
          // When formatting multiple salaries with the same duration
          const formattedSalaries = salaries.map(salary => 
            formatSalaryWithDuration(salary, salary, currency, duration)
          );
          
          // Then all should use the same duration text
          const expectedDurationText = durationText[duration];
          
          formattedSalaries.forEach(formatted => {
            expect(formatted).toContain(expectedDurationText);
          });
        }
      ),
      { numRuns: 100 }
    );
  });
});