import * as fc from 'fast-check';

/**
 * Feature: job-salary-enhancement, Property 3: Duration type storage consistency
 * Validates: Requirements 1.4
 */

describe('Property 3: Duration type storage consistency', () => {
  // Supported duration types as defined in the system
  const supportedDurations = ['hourly', 'daily', 'weekly', 'monthly', 'annually'];

  // Utility function to simulate duration type storage and retrieval
  const storeDurationType = (durationType: string): string => {
    // Simulate the storage process - in real implementation this would be handled by the database
    return durationType;
  };

  it('should store and retrieve the exact same duration type that was selected', async () => {
    await fc.assert(
      fc.property(
        fc.constantFrom(...supportedDurations),
        (selectedDuration) => {
          // When a supported duration is selected and stored
          const storedDuration = storeDurationType(selectedDuration);
          
          // Then the stored duration should match exactly the selected duration
          expect(storedDuration).toBe(selectedDuration);
          
          // And it should be a valid duration from our supported list
          expect(supportedDurations).toContain(storedDuration);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should maintain duration type format consistency', async () => {
    await fc.assert(
      fc.property(
        fc.constantFrom(...supportedDurations),
        (selectedDuration) => {
          const storedDuration = storeDurationType(selectedDuration);
          
          // Duration types should be lowercase
          expect(storedDuration).toBe(storedDuration.toLowerCase());
          
          // Duration types should contain only letters
          expect(storedDuration).toMatch(/^[a-z]+$/);
          
          // Duration types should be one of the expected values
          expect(supportedDurations).toContain(storedDuration);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should handle all supported duration types consistently', () => {
    // Test that each supported duration can be stored and retrieved correctly
    supportedDurations.forEach(duration => {
      const storedDuration = storeDurationType(duration);
      expect(storedDuration).toBe(duration);
    });
  });

  it('should validate duration type semantics', async () => {
    await fc.assert(
      fc.property(
        fc.constantFrom(...supportedDurations),
        (selectedDuration) => {
          const storedDuration = storeDurationType(selectedDuration);
          
          // Each duration should represent a valid time period
          const validTimeUnits = ['hourly', 'daily', 'weekly', 'monthly', 'annually'];
          expect(validTimeUnits).toContain(storedDuration);
          
          // Duration should end with 'ly' (except for some edge cases we might add later)
          expect(storedDuration.endsWith('ly')).toBe(true);
        }
      ),
      { numRuns: 100 }
    );
  });
});