import { validate } from 'class-validator';
import { CreateJobDto } from './create-job.dto';

describe('CreateJobDto Salary Validation', () => {
  it('should pass validation when minSalary <= maxSalary', async () => {
    const dto = new CreateJobDto();
    dto.title = 'Test Job';
    dto.description = 'Test Description';
    dto.location = 'Test Location';
    dto.jobType = 'full-time';
    dto.minSalary = 50000;
    dto.maxSalary = 80000;
    dto.currency = 'USD';
    dto.salaryDuration = 'annually';

    const errors = await validate(dto);
    expect(errors.length).toBe(0);
  });

  it('should fail validation when minSalary > maxSalary', async () => {
    const dto = new CreateJobDto();
    dto.title = 'Test Job';
    dto.description = 'Test Description';
    dto.location = 'Test Location';
    dto.jobType = 'full-time';
    dto.minSalary = 80000;
    dto.maxSalary = 50000;
    dto.currency = 'USD';
    dto.salaryDuration = 'annually';

    const errors = await validate(dto);
    expect(errors.length).toBeGreaterThan(0);
    expect(errors.some(error => 
      error.constraints && 
      Object.values(error.constraints).some(message => 
        message.includes('Maximum salary must be greater than or equal to minimum salary')
      )
    )).toBe(true);
  });

  it('should pass validation when only minSalary is provided', async () => {
    const dto = new CreateJobDto();
    dto.title = 'Test Job';
    dto.description = 'Test Description';
    dto.location = 'Test Location';
    dto.jobType = 'full-time';
    dto.minSalary = 50000;
    dto.currency = 'USD';
    dto.salaryDuration = 'annually';

    const errors = await validate(dto);
    expect(errors.length).toBe(0);
  });

  it('should pass validation when only maxSalary is provided', async () => {
    const dto = new CreateJobDto();
    dto.title = 'Test Job';
    dto.description = 'Test Description';
    dto.location = 'Test Location';
    dto.jobType = 'full-time';
    dto.maxSalary = 80000;
    dto.currency = 'USD';
    dto.salaryDuration = 'annually';

    const errors = await validate(dto);
    expect(errors.length).toBe(0);
  });

  it('should pass validation when no salary fields are provided', async () => {
    const dto = new CreateJobDto();
    dto.title = 'Test Job';
    dto.description = 'Test Description';
    dto.location = 'Test Location';
    dto.jobType = 'full-time';

    const errors = await validate(dto);
    expect(errors.length).toBe(0);
  });

  it('should pass validation with NGN currency', async () => {
    const dto = new CreateJobDto();
    dto.title = 'Test Welder Job';
    dto.description = 'Test job description for welder position';
    dto.location = 'Lagos, Nigeria';
    dto.jobType = 'full-time';
    dto.currency = 'NGN';
    dto.minSalary = 400000;
    dto.maxSalary = 700000;
    dto.salaryDuration = 'monthly';
    dto.requirements = 'Test requirements for welder position';
    dto.isActive = true;

    const errors = await validate(dto);
    expect(errors.length).toBe(0);
  });

  it('should fail validation with invalid currency', async () => {
    const dto = new CreateJobDto();
    dto.title = 'Test Job';
    dto.description = 'Test Description';
    dto.location = 'Test Location';
    dto.jobType = 'full-time';
    dto.currency = 'INVALID_CURRENCY';
    dto.minSalary = 50000;
    dto.maxSalary = 80000;
    dto.salaryDuration = 'annually';

    const errors = await validate(dto);
    expect(errors.length).toBeGreaterThan(0);
    expect(errors.some(error => 
      error.property === 'currency' &&
      error.constraints &&
      Object.values(error.constraints).some(message => 
        message.includes('must be one of the following values')
      )
    )).toBe(true);
  });
});