import * as fc from 'fast-check';

/**
 * Feature: job-salary-enhancement, Property 7: Currency support completeness
 * Validates: Requirements 3.1
 */

describe('Property 7: Currency support completeness', () => {
  // Major international currencies that should be supported
  const supportedCurrencies = ['USD', 'EUR', 'GBP', 'CAD', 'AUD', 'JPY', 'CHF', 'SEK', 'NOK', 'DKK'];
  
  // Currency symbols mapping (matches the service implementation)
  const currencySymbols: Record<string, string> = {
    'USD': '$',
    'EUR': '€',
    'GBP': '£',
    'CAD': 'C$',
    'AUD': 'A$',
    'JPY': '¥',
    'CHF': 'CHF',
    'SEK': 'kr',
    'NOK': 'kr',
    'DKK': 'kr'
  };

  // Utility function to format currency (matches the service implementation)
  const formatCurrency = (amount: number, currencyCode: string): string => {
    const currencyRules: Record<string, {
      symbol: string;
      decimals: number;
      symbolPosition: 'before' | 'after';
    }> = {
      'USD': { symbol: '$', decimals: 2, symbolPosition: 'before' },
      'EUR': { symbol: '€', decimals: 2, symbolPosition: 'before' },
      'GBP': { symbol: '£', decimals: 2, symbolPosition: 'before' },
      'CAD': { symbol: 'C$', decimals: 2, symbolPosition: 'before' },
      'AUD': { symbol: 'A$', decimals: 2, symbolPosition: 'before' },
      'JPY': { symbol: '¥', decimals: 0, symbolPosition: 'before' },
      'CHF': { symbol: 'CHF', decimals: 2, symbolPosition: 'before' },
      'SEK': { symbol: 'kr', decimals: 2, symbolPosition: 'after' },
      'NOK': { symbol: 'kr', decimals: 2, symbolPosition: 'after' },
      'DKK': { symbol: 'kr', decimals: 2, symbolPosition: 'after' }
    };

    const rules = currencyRules[currencyCode];
    if (!rules) {
      throw new Error(`Unsupported currency: ${currencyCode}`);
    }

    const formattedNumber = amount.toLocaleString('en-US', {
      minimumFractionDigits: rules.decimals,
      maximumFractionDigits: rules.decimals,
      useGrouping: true
    });

    if (rules.symbolPosition === 'before') {
      return `${rules.symbol}${formattedNumber}`;
    } else {
      return `${formattedNumber} ${rules.symbol}`;
    }
  };

  it('should accept and process all major international currencies correctly', async () => {
    await fc.assert(
      fc.property(
        fc.constantFrom(...supportedCurrencies),
        fc.float({ min: 1, max: 1000000, noNaN: true }),
        (currency, amount) => {
          // When any supported currency is used
          // Then the system should accept and process it without errors
          expect(() => formatCurrency(amount, currency)).not.toThrow();
          
          const formatted = formatCurrency(amount, currency);
          
          // And should produce a non-empty formatted string
          expect(formatted).toBeTruthy();
          expect(typeof formatted).toBe('string');
          expect(formatted.length).toBeGreaterThan(0);
          
          // And should contain the expected currency symbol
          const expectedSymbol = currencySymbols[currency];
          expect(formatted).toContain(expectedSymbol);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should handle currency-specific formatting rules correctly', async () => {
    await fc.assert(
      fc.property(
        fc.constantFrom(...supportedCurrencies),
        fc.float({ min: 1, max: 100000, noNaN: true }),
        (currency, amount) => {
          // When formatting amounts in different currencies
          const formatted = formatCurrency(amount, currency);
          
          // Then JPY should not have decimal places
          if (currency === 'JPY') {
            expect(formatted).not.toMatch(/¥.*\.\d+/);
            expect(formatted).toMatch(/¥\d+/);
          }
          
          // And Scandinavian currencies should have symbol after the amount
          if (['SEK', 'NOK', 'DKK'].includes(currency)) {
            expect(formatted).toMatch(/\d+.*kr$/);
            expect(formatted).not.toMatch(/^kr/);
          }
          
          // And other currencies should have symbol before the amount
          if (['USD', 'EUR', 'GBP', 'CAD', 'AUD', 'CHF'].includes(currency)) {
            const symbol = currencySymbols[currency];
            expect(formatted).toMatch(new RegExp(`^${symbol.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}`));
          }
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should maintain consistent currency symbols across all supported currencies', async () => {
    await fc.assert(
      fc.property(
        fc.constantFrom(...supportedCurrencies),
        fc.array(fc.float({ min: 1, max: 50000, noNaN: true }), { minLength: 2, maxLength: 5 }),
        (currency, amounts) => {
          // When formatting multiple amounts in the same currency
          const formattedAmounts = amounts.map(amount => formatCurrency(amount, currency));
          
          // Then all should use the same currency symbol
          const expectedSymbol = currencySymbols[currency];
          formattedAmounts.forEach(formatted => {
            expect(formatted).toContain(expectedSymbol);
          });
          
          // And symbol should appear in consistent position for the currency
          const symbolRegex = new RegExp(expectedSymbol.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'));
          const symbolPositions = formattedAmounts.map(formatted => {
            const match = formatted.match(symbolRegex);
            return match ? match.index : -1;
          });
          
          // For currencies with symbol before (most currencies)
          if (['USD', 'EUR', 'GBP', 'CAD', 'AUD', 'JPY', 'CHF'].includes(currency)) {
            symbolPositions.forEach(position => {
              expect(position).toBe(0); // Symbol should be at the start
            });
          }
          
          // For currencies with symbol after (Scandinavian currencies)
          if (['SEK', 'NOK', 'DKK'].includes(currency)) {
            symbolPositions.forEach((position, index) => {
              expect(position).toBeGreaterThan(0); // Symbol should not be at the start
              expect(position).toBe(formattedAmounts[index].length - expectedSymbol.length); // Should be at the end
            });
          }
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should handle decimal formatting correctly for each currency', async () => {
    await fc.assert(
      fc.property(
        fc.constantFrom(...supportedCurrencies),
        fc.float({ min: Math.fround(1.01), max: Math.fround(999.99), noNaN: true }),
        (currency, amount) => {
          // When formatting amounts with decimal places
          const formatted = formatCurrency(amount, currency);
          
          // Then JPY should round to whole numbers (no decimals)
          if (currency === 'JPY') {
            expect(formatted).not.toMatch(/\.\d+/);
            const numericPart = formatted.replace(/[^\d]/g, '');
            expect(parseInt(numericPart)).toBe(Math.round(amount));
          }
          
          // And other currencies should preserve decimal places
          if (currency !== 'JPY') {
            expect(formatted).toMatch(/\.\d{2}/); // Should have exactly 2 decimal places
          }
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should handle large amounts correctly for all currencies', async () => {
    await fc.assert(
      fc.property(
        fc.constantFrom(...supportedCurrencies),
        fc.float({ min: Math.fround(1000000), max: Math.fround(999999999), noNaN: true }),
        (currency, amount) => {
          // When formatting large amounts
          const formatted = formatCurrency(amount, currency);
          
          // Then the system should handle it without errors
          expect(formatted).toBeTruthy();
          expect(typeof formatted).toBe('string');
          
          // And should include thousands separators for readability
          expect(formatted).toMatch(/\d{1,3}(,\d{3})*/);
          
          // And should contain the correct currency symbol
          const expectedSymbol = currencySymbols[currency];
          expect(formatted).toContain(expectedSymbol);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should reject unsupported currencies with clear error messages', async () => {
    await fc.assert(
      fc.property(
        fc.string({ minLength: 3, maxLength: 3 }).filter(code => 
          !supportedCurrencies.includes(code) && 
          /^[A-Z]{3}$/.test(code) // Valid currency code format but not supported
        ),
        fc.float({ min: 1, max: 1000, noNaN: true }),
        (unsupportedCurrency, amount) => {
          // When an unsupported currency is used
          // Then the system should throw a clear error
          expect(() => formatCurrency(amount, unsupportedCurrency)).toThrow();
          expect(() => formatCurrency(amount, unsupportedCurrency)).toThrow(/Unsupported currency/);
        }
      ),
      { numRuns: 50 }
    );
  });

  it('should maintain currency support completeness across the entire supported list', () => {
    // When checking all supported currencies
    // Then each currency should have proper configuration
    supportedCurrencies.forEach(currency => {
      // Should have a symbol mapping
      expect(currencySymbols[currency]).toBeDefined();
      expect(typeof currencySymbols[currency]).toBe('string');
      expect(currencySymbols[currency].length).toBeGreaterThan(0);
      
      // Should be processable by the formatting function
      expect(() => formatCurrency(100, currency)).not.toThrow();
      
      const formatted = formatCurrency(100, currency);
      expect(formatted).toBeTruthy();
      expect(formatted).toContain(currencySymbols[currency]);
    });
    
    // And the list should include all major international currencies as specified
    const expectedMajorCurrencies = ['USD', 'EUR', 'GBP', 'CAD', 'AUD', 'JPY'];
    expectedMajorCurrencies.forEach(currency => {
      expect(supportedCurrencies).toContain(currency);
    });
  });
});