import * as fc from 'fast-check';

/**
 * Feature: job-salary-enhancement, Property 5: Currency formatting consistency
 * Validates: Requirements 2.3
 */

describe('Property 5: Currency formatting consistency', () => {
  // Supported currency codes and their formatting rules
  const supportedCurrencies = ['USD', 'EUR', 'GBP', 'CAD', 'AUD', 'JPY', 'CHF', 'SEK', 'NOK', 'DKK'];

  // Currency formatting rules based on standard conventions
  const currencyRules: Record<string, {
    symbol: string;
    decimals: number;
    symbolPosition: 'before' | 'after';
    thousandsSeparator: string;
    decimalSeparator: string;
  }> = {
    'USD': { symbol: '$', decimals: 2, symbolPosition: 'before', thousandsSeparator: ',', decimalSeparator: '.' },
    'EUR': { symbol: '€', decimals: 2, symbolPosition: 'before', thousandsSeparator: ',', decimalSeparator: '.' },
    'GBP': { symbol: '£', decimals: 2, symbolPosition: 'before', thousandsSeparator: ',', decimalSeparator: '.' },
    'CAD': { symbol: 'C$', decimals: 2, symbolPosition: 'before', thousandsSeparator: ',', decimalSeparator: '.' },
    'AUD': { symbol: 'A$', decimals: 2, symbolPosition: 'before', thousandsSeparator: ',', decimalSeparator: '.' },
    'JPY': { symbol: '¥', decimals: 0, symbolPosition: 'before', thousandsSeparator: ',', decimalSeparator: '.' },
    'CHF': { symbol: 'CHF', decimals: 2, symbolPosition: 'before', thousandsSeparator: ',', decimalSeparator: '.' },
    'SEK': { symbol: 'kr', decimals: 2, symbolPosition: 'after', thousandsSeparator: ',', decimalSeparator: '.' },
    'NOK': { symbol: 'kr', decimals: 2, symbolPosition: 'after', thousandsSeparator: ',', decimalSeparator: '.' },
    'DKK': { symbol: 'kr', decimals: 2, symbolPosition: 'after', thousandsSeparator: ',', decimalSeparator: '.' }
  };

  // Utility function to format currency according to standard rules
  const formatCurrency = (amount: number, currencyCode: string): string => {
    const rules = currencyRules[currencyCode];
    if (!rules) {
      throw new Error(`Unsupported currency: ${currencyCode}`);
    }

    // Format the number with appropriate decimal places
    const formattedNumber = amount.toLocaleString('en-US', {
      minimumFractionDigits: rules.decimals,
      maximumFractionDigits: rules.decimals,
      useGrouping: true
    });

    // Apply symbol position
    if (rules.symbolPosition === 'before') {
      return `${rules.symbol}${formattedNumber}`;
    } else {
      return `${formattedNumber} ${rules.symbol}`;
    }
  };

  it('should format currency according to standard currency display rules', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: Math.fround(0.01), max: Math.fround(1000000), noNaN: true }),
        fc.constantFrom(...supportedCurrencies),
        (amount, currencyCode) => {
          // When formatting a currency amount
          const formatted = formatCurrency(amount, currencyCode);
          const rules = currencyRules[currencyCode];
          
          // Then the result should contain the correct currency symbol
          expect(formatted).toContain(rules.symbol);
          
          // And should follow the correct symbol position
          if (rules.symbolPosition === 'before') {
            expect(formatted).toMatch(new RegExp(`^${rules.symbol.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}`));
          } else {
            expect(formatted).toMatch(new RegExp(`${rules.symbol.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}$`));
          }
          
          // And should contain numeric content
          expect(formatted).toMatch(/\d/);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should apply correct decimal places for each currency', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: Math.fround(0.01), max: Math.fround(1000000), noNaN: true }),
        fc.constantFrom(...supportedCurrencies),
        (amount, currencyCode) => {
          const formatted = formatCurrency(amount, currencyCode);
          const rules = currencyRules[currencyCode];
          
          if (rules.decimals === 0) {
            // For currencies with no decimals (like JPY), should not contain decimal point
            expect(formatted).not.toMatch(/\.\d+/);
          } else {
            // For currencies with decimals, should contain decimal point with correct number of places
            const decimalMatch = formatted.match(/\.(\d+)/);
            if (decimalMatch) {
              expect(decimalMatch[1]).toHaveLength(rules.decimals);
            }
          }
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should handle large numbers with thousands separators consistently', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: Math.fround(1000), max: Math.fround(10000000), noNaN: true }),
        fc.constantFrom(...supportedCurrencies),
        (amount, currencyCode) => {
          const formatted = formatCurrency(amount, currencyCode);
          
          // For amounts >= 1000, should contain thousands separator
          if (amount >= 1000) {
            expect(formatted).toMatch(/,/);
          }
          
          // Should not have leading zeros in the main number
          expect(formatted).not.toMatch(/[^0-9]0\d{4,}/);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should produce consistent formatting for the same currency and amount', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: Math.fround(0.01), max: Math.fround(1000000), noNaN: true }),
        fc.constantFrom(...supportedCurrencies),
        (amount, currencyCode) => {
          // When the same amount and currency are formatted multiple times
          const formatted1 = formatCurrency(amount, currencyCode);
          const formatted2 = formatCurrency(amount, currencyCode);
          
          // Then the results should be identical
          expect(formatted1).toBe(formatted2);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should handle special formatting rules for Scandinavian currencies', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: Math.fround(0.01), max: Math.fround(1000000), noNaN: true }),
        fc.constantFrom('SEK', 'NOK', 'DKK'),
        (amount, currencyCode) => {
          const formatted = formatCurrency(amount, currencyCode);
          
          // Scandinavian currencies should have symbol after the amount
          expect(formatted).toMatch(/\d+ kr$/);
          
          // Should contain the 'kr' symbol
          expect(formatted).toContain('kr');
          
          // Should not start with 'kr'
          expect(formatted).not.toMatch(/^kr/);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should handle Japanese Yen special formatting (no decimals)', async () => {
    await fc.assert(
      fc.property(
        fc.float({ min: Math.fround(1), max: Math.fround(1000000), noNaN: true }),
        (amount) => {
          const formatted = formatCurrency(amount, 'JPY');
          
          // JPY should not have decimal places
          expect(formatted).not.toMatch(/\.\d+/);
          
          // Should start with yen symbol
          expect(formatted).toMatch(/^¥/);
          
          // Should contain only whole numbers after the symbol
          expect(formatted).toMatch(/^¥\d{1,3}(,\d{3})*$/);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should validate all supported currencies have proper formatting rules', () => {
    // Test that each supported currency has defined formatting rules
    supportedCurrencies.forEach(currency => {
      expect(currencyRules).toHaveProperty(currency);
      
      const rules = currencyRules[currency];
      expect(rules.symbol).toBeDefined();
      expect(typeof rules.decimals).toBe('number');
      expect(rules.decimals).toBeGreaterThanOrEqual(0);
      expect(['before', 'after']).toContain(rules.symbolPosition);
    });
  });
});