import { Test, TestingModule } from '@nestjs/testing';
import { ExecutionContext, UnauthorizedException } from '@nestjs/common';
import { JwtAuthGuard } from './jwt-auth.guard';
import * as fc from 'fast-check';

/**
 * Feature: maritime-app-enhancements, Property 32: Unauthenticated requests are rejected
 * Validates: Requirements 10.5
 * 
 * For any protected API endpoint, requests without valid authentication 
 * should be rejected with 401 status code
 */

describe('JwtAuthGuard Property Tests', () => {
  let guard: JwtAuthGuard;

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [JwtAuthGuard]
    }).compile();

    guard = module.get<JwtAuthGuard>(JwtAuthGuard);
  });

  describe('Property 32: Unauthenticated requests are rejected', () => {
    it('should reject requests without authentication token', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.record({
            headers: fc.record({
              authorization: fc.oneof(
                fc.constant(undefined),
                fc.constant(''),
                fc.constant('InvalidToken'),
                fc.string().filter(s => !s.startsWith('Bearer '))
              )
            })
          }),
          async (requestData) => {
            const mockContext = {
              switchToHttp: () => ({
                getRequest: () => requestData
              })
            } as ExecutionContext;

            try {
              await guard.canActivate(mockContext);
              // If no error is thrown, the request should have valid auth
              // In this test, we're testing invalid auth, so this shouldn't happen
              expect(requestData.headers.authorization).toBeDefined();
            } catch (error) {
              // Expect UnauthorizedException for invalid/missing auth
              expect(error).toBeDefined();
            }
          }
        ),
        { numRuns: 100 }
      );
    });
  });
});
