import { Injectable, Logger } from '@nestjs/common';
import * as emailjs from '@emailjs/nodejs';

@Injectable()
export class EmailService {
  private readonly logger = new Logger(EmailService.name);
  
  // EmailJS configuration - these should be set in environment variables
  private readonly serviceId = process.env.EMAILJS_SERVICE_ID || 'your_service_id';
  private readonly publicKey = process.env.EMAILJS_PUBLIC_KEY || 'your_public_key';
  private readonly privateKey = process.env.EMAILJS_PRIVATE_KEY || 'your_private_key';
  private readonly verificationTemplateId = process.env.EMAILJS_VERIFICATION_TEMPLATE_ID || 'template_verification';
  private readonly resetTemplateId = process.env.EMAILJS_RESET_TEMPLATE_ID || 'template_reset';

  constructor() {
    // Initialize EmailJS with your keys
    if (this.privateKey !== 'your_private_key') {
      this.logger.log('Initializing EmailJS with provided credentials...');
      emailjs.init({
        publicKey: this.publicKey,
        privateKey: this.privateKey,
      });
      this.logger.log(`EmailJS initialized - Service: ${this.serviceId}, Templates: ${this.verificationTemplateId}, ${this.resetTemplateId}`);
    } else {
      this.logger.warn('EmailJS credentials not found in environment variables. Running in development mode.');
    }
  }

  async sendVerificationEmail(email: string, otp: string): Promise<void> {
    try {
      // For development, log the OTP
      this.logger.log(`Sending verification email to ${email} with OTP: ${otp}`);
      
      if (this.privateKey === 'your_private_key') {
        this.logger.warn('EmailJS not configured. Using development mode.');
        // Simulate email sending delay in development
        await new Promise(resolve => setTimeout(resolve, 1000));
        return;
      }

      this.logger.log(`Attempting to send verification email via EmailJS to ${email}...`);
      this.logger.debug(`Using service: ${this.serviceId}, template: ${this.verificationTemplateId}`);

      const templateParams = {
        to_email: email,
        to_name: email.split('@')[0],
        otp_code: otp,
        app_name: 'Linked Blue Ocean',
        expires_in: '10 minutes',
        support_email: 'support@linkedblueocean.com'
      };

      this.logger.debug('Template params:', templateParams);

      const response = await emailjs.send(
        this.serviceId,
        this.verificationTemplateId,
        templateParams
      );

      this.logger.log(`Verification email sent successfully to ${email}. Response:`, response);
    } catch (error) {
      this.logger.error(`Failed to send verification email to ${email}:`, error);
      this.logger.warn(`EmailJS failed, falling back to development mode. OTP for ${email}: ${otp}`);
      // Don't throw error - allow registration to continue in development mode
      // In production, you might want to throw the error or implement retry logic
    }
  }

  async sendPasswordResetEmail(email: string, otp: string): Promise<void> {
    try {
      // For development, log the OTP
      this.logger.log(`Sending password reset email to ${email} with OTP: ${otp}`);
      
      if (this.privateKey === 'your_private_key') {
        this.logger.warn('EmailJS not configured. Using development mode.');
        // Simulate email sending delay in development
        await new Promise(resolve => setTimeout(resolve, 1000));
        return;
      }

      this.logger.log(`Attempting to send password reset email via EmailJS to ${email}...`);
      this.logger.debug(`Using service: ${this.serviceId}, template: ${this.resetTemplateId}`);

      const templateParams = {
        to_email: email,
        to_name: email.split('@')[0],
        otp_code: otp,
        app_name: 'Linked Blue Ocean',
        expires_in: '10 minutes',
        support_email: 'support@linkedblueocean.com'
      };

      this.logger.debug('Template params:', templateParams);

      const response = await emailjs.send(
        this.serviceId,
        this.resetTemplateId,
        templateParams
      );

      this.logger.log(`Password reset email sent successfully to ${email}. Response:`, response);
    } catch (error) {
      this.logger.error(`Failed to send password reset email to ${email}:`, error);
      this.logger.warn(`EmailJS failed, falling back to development mode. OTP for ${email}: ${otp}`);
      // Don't throw error - allow password reset to continue in development mode
      // In production, you might want to throw the error or implement retry logic
    }
  }
}