import { DataSource } from 'typeorm';
import { JobEntity } from '../src/jobs/job.entity';

/**
 * Migration Validation Script
 * 
 * This script validates that the salary enhancement migration
 * has been applied correctly and data integrity is maintained.
 */

interface ValidationResult {
  success: boolean;
  message: string;
  details?: any;
}

class MigrationValidator {
  private dataSource: DataSource;

  constructor(dataSource: DataSource) {
    this.dataSource = dataSource;
  }

  async validateSchema(): Promise<ValidationResult> {
    try {
      const queryRunner = this.dataSource.createQueryRunner();
      
      // Check if new columns exist
      const columns = await queryRunner.getTable('jobs');
      const columnNames = columns?.columns.map(col => col.name) || [];
      
      const requiredColumns = ['minSalary', 'maxSalary', 'currency', 'salaryDuration'];
      const missingColumns = requiredColumns.filter(col => !columnNames.includes(col));
      
      if (missingColumns.length > 0) {
        return {
          success: false,
          message: `Missing required columns: ${missingColumns.join(', ')}`,
          details: { missingColumns, existingColumns: columnNames }
        };
      }

      // Validate column types and constraints
      const salaryColumns = columns?.columns.filter(col => 
        ['minSalary', 'maxSalary'].includes(col.name)
      );
      
      for (const col of salaryColumns || []) {
        if (col.type !== 'numeric' && col.type !== 'decimal') {
          return {
            success: false,
            message: `Invalid column type for ${col.name}: expected decimal, got ${col.type}`,
            details: { column: col.name, type: col.type }
          };
        }
      }

      await queryRunner.release();
      
      return {
        success: true,
        message: 'Schema validation passed - all required columns exist with correct types'
      };
      
    } catch (error) {
      return {
        success: false,
        message: `Schema validation failed: ${error instanceof Error ? error.message : String(error)}`,
        details: error
      };
    }
  }

  async validateDataIntegrity(): Promise<ValidationResult> {
    try {
      const jobRepository = this.dataSource.getRepository(JobEntity);
      
      // Check that existing jobs still have their data
      const totalJobs = await jobRepository.count();
      const jobsWithLegacySalary = await jobRepository.count({
        where: { salaryRange: { $ne: null } as any }
      });
      
      // Validate salary range constraints
      const invalidSalaryRanges = await jobRepository
        .createQueryBuilder('job')
        .where('job.minSalary IS NOT NULL')
        .andWhere('job.maxSalary IS NOT NULL')
        .andWhere('job.minSalary > job.maxSalary')
        .getCount();
      
      if (invalidSalaryRanges > 0) {
        return {
          success: false,
          message: `Found ${invalidSalaryRanges} jobs with invalid salary ranges (min > max)`,
          details: { invalidRanges: invalidSalaryRanges }
        };
      }

      return {
        success: true,
        message: 'Data integrity validation passed',
        details: {
          totalJobs,
          jobsWithLegacySalary,
          invalidSalaryRanges
        }
      };
      
    } catch (error) {
      return {
        success: false,
        message: `Data integrity validation failed: ${error instanceof Error ? error.message : String(error)}`,
        details: error
      };
    }
  }

  async validateEnumConstraints(): Promise<ValidationResult> {
    try {
      const queryRunner = this.dataSource.createQueryRunner();
      
      // Test currency enum constraint
      try {
        await queryRunner.query(`
          INSERT INTO jobs (id, title, description, location, "jobType", requirements, currency, "companyId")
          VALUES ('test-currency-uuid', 'Test Job', 'Test Description', 'Test Location', 'full-time', 'Test Requirements', 'INVALID_CURRENCY', 'test-company-uuid')
        `);
        
        // If we get here, the constraint failed
        await queryRunner.query(`DELETE FROM jobs WHERE id = 'test-currency-uuid'`);
        
        return {
          success: false,
          message: 'Currency enum constraint is not working - invalid currency was accepted',
          details: { constraint: 'currency enum' }
        };
        
      } catch (error) {
        // This is expected - the constraint should reject invalid values
        if (error instanceof Error && error.message.includes('invalid input value for enum')) {
          // Test passed - constraint is working
        } else {
          throw error; // Unexpected error
        }
      }

      // Test duration enum constraint
      try {
        await queryRunner.query(`
          INSERT INTO jobs (id, title, description, location, "jobType", requirements, "salaryDuration", "companyId")
          VALUES ('test-duration-uuid', 'Test Job', 'Test Description', 'Test Location', 'full-time', 'Test Requirements', 'invalid_duration', 'test-company-uuid')
        `);
        
        // If we get here, the constraint failed
        await queryRunner.query(`DELETE FROM jobs WHERE id = 'test-duration-uuid'`);
        
        return {
          success: false,
          message: 'Duration enum constraint is not working - invalid duration was accepted',
          details: { constraint: 'salaryDuration enum' }
        };
        
      } catch (error) {
        // This is expected - the constraint should reject invalid values
        if (error instanceof Error && error.message.includes('invalid input value for enum')) {
          // Test passed - constraint is working
        } else {
          throw error; // Unexpected error
        }
      }

      await queryRunner.release();
      
      return {
        success: true,
        message: 'Enum constraints validation passed - invalid values are properly rejected'
      };
      
    } catch (error) {
      return {
        success: false,
        message: `Enum constraints validation failed: ${error instanceof Error ? (error as Error).message : String(error)}`,
        details: error
      };
    }
  }

  async runAllValidations(): Promise<ValidationResult[]> {
    console.log('🔍 Running migration validation tests...\n');
    
    const results: ValidationResult[] = [];
    
    // Schema validation
    console.log('📋 Validating database schema...');
    const schemaResult = await this.validateSchema();
    results.push(schemaResult);
    console.log(schemaResult.success ? '✅' : '❌', schemaResult.message);
    
    // Data integrity validation
    console.log('\n📊 Validating data integrity...');
    const dataResult = await this.validateDataIntegrity();
    results.push(dataResult);
    console.log(dataResult.success ? '✅' : '❌', dataResult.message);
    if (dataResult.details) {
      console.log('   Details:', dataResult.details);
    }
    
    // Enum constraints validation
    console.log('\n🔒 Validating enum constraints...');
    const enumResult = await this.validateEnumConstraints();
    results.push(enumResult);
    console.log(enumResult.success ? '✅' : '❌', enumResult.message);
    
    return results;
  }
}

// Export for use in other scripts
export { MigrationValidator, ValidationResult };

// CLI execution
if (require.main === module) {
  async function main() {
    const { AppDataSource } = await import('../src/data-source');
    
    try {
      await AppDataSource.initialize();
      console.log('📡 Connected to database\n');
      
      const validator = new MigrationValidator(AppDataSource);
      const results = await validator.runAllValidations();
      
      const allPassed = results.every(result => result.success);
      
      console.log('\n' + '='.repeat(50));
      console.log(allPassed ? '🎉 All validations passed!' : '💥 Some validations failed!');
      console.log('='.repeat(50));
      
      if (!allPassed) {
        const failures = results.filter(r => !r.success);
        console.log('\n❌ Failed validations:');
        failures.forEach(failure => {
          console.log(`   - ${failure.message}`);
        });
        process.exit(1);
      }
      
    } catch (error) {
      console.error('💥 Validation script failed:', error instanceof Error ? error.message : String(error));
      process.exit(1);
    } finally {
      await AppDataSource.destroy();
    }
  }
  
  main();
}