"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.send = void 0;
const store_js_1 = require("../../store/store.js");
const sendPost_js_1 = require("../../api/sendPost.js");
const validateParams_js_1 = require("../../utils/validateParams/validateParams.js");
const validateTemplateParams_js_1 = require("../../utils/validateTemplateParams/validateTemplateParams.js");
const isBlockedValueInParams_js_1 = require("../../utils/isBlockedValueInParams/isBlockedValueInParams.js");
const blockedEmailError_js_1 = require("../../errors/blockedEmailError/blockedEmailError.js");
const isLimitRateHit_js_1 = require("../../utils/isLimitRateHit/isLimitRateHit.js");
const limitRateError_js_1 = require("../../errors/limitRateError/limitRateError.js");
/**
 * Send a template to the specific EmailJS service
 * @param {string} serviceID - the EmailJS service ID
 * @param {string} templateID - the EmailJS template ID
 * @param {object} templateParams - the template params, what will be set to the EmailJS template
 * @param {object} options - the EmailJS SDK config options
 * @returns {Promise<EmailJSResponseStatus>}
 */
const send = async (serviceID, templateID, templateParams, options) => {
    const publicKey = (options === null || options === void 0 ? void 0 : options.publicKey) || store_js_1.store.publicKey;
    const privateKey = (options === null || options === void 0 ? void 0 : options.privateKey) || store_js_1.store.privateKey;
    const storageProvider = (options === null || options === void 0 ? void 0 : options.storageProvider) || store_js_1.store.storageProvider;
    const blockList = { ...store_js_1.store.blockList, ...options === null || options === void 0 ? void 0 : options.blockList };
    const limitRate = { ...store_js_1.store.limitRate, ...options === null || options === void 0 ? void 0 : options.limitRate };
    (0, validateParams_js_1.validateParams)(publicKey, serviceID, templateID);
    (0, validateTemplateParams_js_1.validateTemplateParams)(templateParams);
    if (templateParams && (0, isBlockedValueInParams_js_1.isBlockedValueInParams)(blockList, templateParams)) {
        return Promise.reject((0, blockedEmailError_js_1.blockedEmailError)());
    }
    if (await (0, isLimitRateHit_js_1.isLimitRateHit)(limitRate, storageProvider)) {
        return Promise.reject((0, limitRateError_js_1.limitRateError)());
    }
    const params = {
        lib_version: '5.0.2',
        user_id: publicKey,
        accessToken: privateKey,
        service_id: serviceID,
        template_id: templateID,
        template_params: templateParams,
    };
    return (0, sendPost_js_1.sendPost)(JSON.stringify(params));
};
exports.send = send;
