#!/bin/bash

# LinkedBlue Backend Deployment Script
# This script prepares and deploys the backend application

echo "🚀 LinkedBlue Backend Deployment Script"
echo "========================================"

# Check if Node.js is installed
if ! command -v node &> /dev/null; then
    echo "❌ Node.js is not installed. Please install Node.js v18+ first."
    exit 1
fi

# Check Node.js version
NODE_VERSION=$(node -v | cut -d'v' -f2 | cut -d'.' -f1)
if [ "$NODE_VERSION" -lt 18 ]; then
    echo "❌ Node.js version 18+ is required. Current version: $(node -v)"
    exit 1
fi

echo "✅ Node.js version: $(node -v)"

# Check if npm is installed
if ! command -v npm &> /dev/null; then
    echo "❌ npm is not installed."
    exit 1
fi

echo "✅ npm version: $(npm -v)"

# Install dependencies
echo "📦 Installing dependencies..."
npm ci --only=production

if [ $? -ne 0 ]; then
    echo "❌ Failed to install dependencies"
    exit 1
fi

echo "✅ Dependencies installed successfully"

# Check if .env file exists
if [ ! -f ".env" ]; then
    echo "⚠️  .env file not found. Please create one based on .env.example"
    echo "📋 Required environment variables:"
    echo "   - PORT"
    echo "   - DB_HOST, DB_PORT, DB_USER, DB_PASSWORD, DB_NAME"
    echo "   - JWT_SECRET, JWT_EXPIRES_IN"
    echo "   - EMAILJS_* variables"
    echo ""
    echo "💡 Copy .env.example to .env and update the values:"
    echo "   cp .env.example .env"
    exit 1
fi

echo "✅ Environment file found"

# Check if dist directory exists
if [ ! -d "dist" ]; then
    echo "❌ Build directory (dist) not found. Please run 'npm run build' first."
    exit 1
fi

echo "✅ Build directory found"

# Check if main.js exists
if [ ! -f "dist/src/main.js" ]; then
    echo "❌ Main application file not found. Please run 'npm run build' first."
    exit 1
fi

echo "✅ Application build verified"

# Create uploads directory if it doesn't exist
mkdir -p uploads/avatars
mkdir -p uploads/cvs
mkdir -p uploads/posts
mkdir -p uploads/news

echo "✅ Upload directories created"

# Set proper permissions for uploads directory
chmod -R 755 uploads

echo "✅ Upload directory permissions set"

# Run database migrations (optional - uncomment if needed)
# echo "🗄️  Running database migrations..."
# npm run migration:run
# 
# if [ $? -ne 0 ]; then
#     echo "⚠️  Database migrations failed. Please check your database connection."
#     echo "   You can run migrations manually with: npm run migration:run"
# else
#     echo "✅ Database migrations completed"
# fi

echo ""
echo "🎉 Deployment preparation completed successfully!"
echo ""
echo "📋 To start the application:"
echo "   npm run start:prod"
echo ""
echo "🌐 The application will be available at:"
echo "   http://localhost:3000 (or your configured PORT)"
echo ""
echo "📚 API Documentation:"
echo "   Check DEPLOYMENT_BUILD_SUMMARY.md for available endpoints"
echo ""
echo "🔧 Troubleshooting:"
echo "   - Check .env file configuration"
echo "   - Ensure PostgreSQL is running"
echo "   - Check logs for any errors"
echo "   - Verify all environment variables are set"